<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';

// Check if user has permission to view system logs
error_log("Checking permissions for user ID: " . $_SESSION['user_id']);
error_log("User role: " . getUserRole());
error_log("User permissions: " . implode(', ', getUserPermissions()));

// Check if user has any system logs permissions
$logsPermissions = [
    'view_logs', 'view_system_logs', 'manage_logs'
];

if (!hasAnyPermission($logsPermissions)) {
    $_SESSION['error'] = "You don't have permission to access system logs.";
    header('Location: dashboard.php');
    exit;
}

if (false) { // Disable the old check
    // Permission modal will be shown instead of redirecting
    error_log("Permission check failed for view_logs");
}

$activePage = 'system_logs';

// Get filter parameters
$module = $_GET['module'] ?? '';
$action_type = $_GET['action_type'] ?? '';
$message_type = $_GET['message_type'] ?? '';
$message_status = $_GET['message_status'] ?? '';
$user_id = $_GET['user_id'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$search = $_GET['search'] ?? '';

// Build query
$query = "
    SELECT l.*, u.username 
    FROM system_logs l 
    LEFT JOIN users u ON l.user_id = u.id 
    WHERE 1=1
";
$params = [];

if ($module) {
    $query .= " AND l.module = ?";
    $params[] = $module;
}

if ($action_type) {
    $query .= " AND l.action_type = ?";
    $params[] = $action_type;
}

if ($message_type) {
    $query .= " AND l.message_type = ?";
    $params[] = $message_type;
}

if ($message_status) {
    $query .= " AND l.message_status = ?";
    $params[] = $message_status;
}

if ($user_id) {
    $query .= " AND l.user_id = ?";
    $params[] = $user_id;
}

if ($date_from) {
    $query .= " AND DATE(l.created_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $query .= " AND DATE(l.created_at) <= ?";
    $params[] = $date_to;
}

if ($search) {
    $query .= " AND (l.details LIKE ? OR l.action LIKE ? OR l.recipient LIKE ? OR u.username LIKE ?)";
    $search_param = "%$search%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
}

// Add SMS/Email/WhatsApp logs filter if requested
$show_messages = isset($_GET['show_messages']) && $_GET['show_messages'] == '1';
if ($show_messages) {
    $query .= " AND l.message_type IS NOT NULL";
}

$query .= " ORDER BY l.created_at DESC";

// Get logs
$stmt = $pdo->prepare($query);
$stmt->execute($params);
$logs = $stmt->fetchAll();

// Get unique values for filters
$modules = $pdo->query("SELECT DISTINCT module FROM system_logs WHERE module IS NOT NULL ORDER BY module")->fetchAll(PDO::FETCH_COLUMN);
$action_types = $pdo->query("SELECT DISTINCT action_type FROM system_logs WHERE action_type IS NOT NULL ORDER BY action_type")->fetchAll(PDO::FETCH_COLUMN);
$message_types = $pdo->query("SELECT DISTINCT message_type FROM system_logs WHERE message_type IS NOT NULL ORDER BY message_type")->fetchAll(PDO::FETCH_COLUMN);
$message_statuses = $pdo->query("SELECT DISTINCT message_status FROM system_logs WHERE message_status IS NOT NULL ORDER BY message_status")->fetchAll(PDO::FETCH_COLUMN);
$users = $pdo->query("SELECT id, username FROM users ORDER BY username")->fetchAll();

// Get message statistics
$messageStats = $pdo->query("
    SELECT 
        message_type,
        message_status,
        COUNT(*) as count
    FROM system_logs 
    WHERE message_type IS NOT NULL 
    GROUP BY message_type, message_status
")->fetchAll();

$stats = [];
foreach ($messageStats as $stat) {
    $stats[$stat['message_type']][$stat['message_status']] = $stat['count'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Logs - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #e0e7ff 0%, #f8fafc 100%);
            font-family: 'Inter', sans-serif;
            min-height: 100vh;
        }
        .card {
            border-radius: 22px;
            box-shadow: 0 8px 32px rgba(37,99,235,0.13), 0 1.5px 6px rgba(0,0,0,0.03);
            border: none;
            background: rgba(255,255,255,0.85);
            backdrop-filter: blur(6px);
        }
        .card-header {
            background: none;
            border-bottom: none;
        }
        .btn-primary, .btn-warning, .btn-info, .btn-danger {
            border-radius: 10px;
            font-weight: 600;
            box-shadow: 0 2px 8px rgba(59,130,246,0.08);
            transition: box-shadow 0.2s;
        }
        .btn-primary:hover, .btn-info:hover {
            box-shadow: 0 4px 16px rgba(59,130,246,0.18);
        }
        .filter-section {
            background: rgba(255,255,255,0.7);
            padding: 1.2rem;
            border-radius: 16px;
            margin-bottom: 1.5rem;
            box-shadow: 0 2px 12px rgba(37,99,235,0.06);
        }
        .timeline-container {
            position: relative;
            padding-left: 2.5rem;
            margin-top: 2rem;
        }
        .timeline-bar {
            width: 6px;
            background: linear-gradient(180deg, #6366f1 0%, #38bdf8 100%);
            border-radius: 3px;
            left: 12px;
            top: 0;
            bottom: 0;
            z-index: 0;
            box-shadow: 0 0 16px #6366f1aa;
        }
        .timeline-entry {
            display: flex;
            margin-bottom: 2.5rem;
            position: relative;
            animation: fadeInUp 0.7s cubic-bezier(.39,.575,.56,1) both;
        }
        @keyframes fadeInUp {
            0% { opacity: 0; transform: translateY(40px); }
            100% { opacity: 1; transform: none; }
        }
        .timeline-dot {
            position: absolute;
            left: 0;
            top: 0;
            transform: translate(-50%, 0);
            width: 38px;
            height: 38px;
            background: linear-gradient(135deg, #6366f1 60%, #38bdf8 100%);
            border: 4px solid #fff;
            box-shadow: 0 4px 16px #6366f1aa, 0 1.5px 6px rgba(0,0,0,0.04);
            border-radius: 50%;
            z-index: 2;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: box-shadow 0.2s, transform 0.2s;
            animation: pulseDot 1.5s infinite alternate;
        }
        @keyframes pulseDot {
            0% { box-shadow: 0 0 0 0 #6366f1aa, 0 1.5px 6px rgba(0,0,0,0.04); }
            100% { box-shadow: 0 0 0 12px #6366f122, 0 1.5px 6px rgba(0,0,0,0.04); }
        }
        .timeline-dot .avatar-initials {
            font-size: 1.3rem;
            font-weight: 700;
            color: #fff;
            text-shadow: 0 2px 8px #6366f1cc;
        }
        .timeline-entry:hover .timeline-dot {
            box-shadow: 0 0 0 18px #38bdf822, 0 4px 16px #6366f1cc;
            transform: translate(-50%, -4px) scale(1.08);
        }
        .timeline-entry .ms-5 {
            margin-left: 2.7rem !important;
        }
        .timeline-entry .card-glass {
            background: rgba(255,255,255,0.85);
            border-radius: 18px;
            box-shadow: 0 2px 16px rgba(59,130,246,0.10);
            padding: 1.1rem 1.3rem 1.1rem 1.3rem;
            transition: box-shadow 0.2s, background 0.2s;
            border: 1.5px solid #e0e7ff;
        }
        .timeline-entry:hover .card-glass {
            background: rgba(236,245,255,0.97);
            box-shadow: 0 6px 32px #38bdf855, 0 2px 8px #6366f1aa;
        }
        .fw-bold.text-dark {
            letter-spacing: 0.01em;
        }
        .badge {
            border-radius: 8px;
            font-size: 0.92rem;
            font-weight: 600;
            padding: 0.5em 1.1em;
            box-shadow: 0 1px 4px #6366f122;
            margin-right: 0.5em;
        }
        .bg-success { background: linear-gradient(90deg,#22c55e 60%,#4ade80 100%)!important; color: #fff!important; }
        .bg-danger { background: linear-gradient(90deg,#ef4444 60%,#f87171 100%)!important; color: #fff!important; }
        .bg-warning { background: linear-gradient(90deg,#f59e42 60%,#fbbf24 100%)!important; color: #fff!important; }
        .bg-info { background: linear-gradient(90deg,#0ea5e9 60%,#38bdf8 100%)!important; color: #fff!important; }
        .bg-primary { background: linear-gradient(90deg,#6366f1 60%,#818cf8 100%)!important; color: #fff!important; }
        .bg-secondary { background: linear-gradient(90deg,#64748b 60%,#a1a1aa 100%)!important; color: #fff!important; }
        .log-details {
            font-size: 1.05rem;
            background: rgba(248,250,252,0.95);
            border-radius: 10px;
            padding: 0.7rem 1rem;
            margin-top: 0.5rem;
            box-shadow: 0 1px 4px #6366f122;
            border: 1px solid #e0e7ff;
            transition: background 0.2s;
        }
        .timeline-entry:hover .log-details {
            background: #f1f5f9;
        }
        .fw-semibold.text-primary, .fw-semibold.text-success {
            font-weight: 700!important;
            letter-spacing: 0.01em;
        }
        .fw-semibold.text-success { color: #22c55e!important; }
        .fw-semibold.text-primary { color: #6366f1!important; }
        .fw-bold.text-dark { color: #22223b!important; }
        .text-muted { color: #64748b!important; }
        @media (max-width: 768px) {
            .timeline-container { padding-left: 1.2rem; }
            .timeline-entry .ms-5 { margin-left: 2.1rem !important; }
            .timeline-dot { width: 28px; height: 28px; font-size: 1rem; }
            .timeline-entry .card-glass { padding: 0.7rem 0.7rem 0.7rem 0.7rem; }
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">System Logs</h4>
                        <a href="?show_messages=1" class="btn btn-info">Messages</a>
                    </div>
                    <div class="card-body">
                        <!-- Filters -->
                        <div class="filter-section mb-4 p-3 rounded shadow-sm bg-white">
                            <form method="get" class="row g-2 align-items-end">
                                <div class="col-md-2">
                                    <label class="form-label">Module</label>
                                    <select name="module" class="form-select form-select-sm">
                                        <option value="">All Modules</option>
                                        <?php foreach ($modules as $m): ?>
                                            <option value="<?php echo $m; ?>" <?php echo $module === $m ? 'selected' : ''; ?>>
                                                <?php echo ucfirst($m); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Action Type</label>
                                    <select name="action_type" class="form-select form-select-sm">
                                        <option value="">All Actions</option>
                                        <?php foreach ($action_types as $at): ?>
                                            <option value="<?php echo $at; ?>" <?php echo $action_type === $at ? 'selected' : ''; ?>>
                                                <?php echo ucwords(str_replace('_', ' ', $at)); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Message Type</label>
                                    <select name="message_type" class="form-select form-select-sm">
                                        <option value="">All Message Types</option>
                                        <?php foreach ($message_types as $mt): ?>
                                            <option value="<?php echo $mt; ?>" <?php echo $message_type === $mt ? 'selected' : ''; ?>>
                                                <?php echo ucwords(str_replace('_', ' ', $mt)); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Message Status</label>
                                    <select name="message_status" class="form-select form-select-sm">
                                        <option value="">All Message Statuses</option>
                                        <?php foreach ($message_statuses as $ms): ?>
                                            <option value="<?php echo $ms; ?>" <?php echo $message_status === $ms ? 'selected' : ''; ?>>
                                                <?php echo ucwords(str_replace('_', ' ', $ms)); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">User</label>
                                    <select name="user_id" class="form-select form-select-sm">
                                        <option value="">All Users</option>
                                        <?php foreach ($users as $u): ?>
                                            <option value="<?php echo $u['id']; ?>" <?php echo $user_id == $u['id'] ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($u['username']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Date From</label>
                                    <input type="date" name="date_from" class="form-control form-control-sm" value="<?php echo $date_from; ?>">
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Date To</label>
                                    <input type="date" name="date_to" class="form-control form-control-sm" value="<?php echo $date_to; ?>">
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Search</label>
                                    <input type="text" name="search" class="form-control form-control-sm" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search logs...">
                                </div>
                                <div class="col-12 text-end mt-2">
                                    <button type="submit" class="btn btn-primary btn-sm">
                                        <i class="fas fa-filter me-2"></i>Apply Filters
                                    </button>
                                    <a href="system_logs.php" class="btn btn-secondary btn-sm">
                                        <i class="fas fa-times me-2"></i>Clear Filters
                                    </a>
                                </div>
                            </form>
                        </div>

                        <!-- Message Statistics (only show when viewing messages) -->
                        <?php if ($show_messages && !empty($stats)): ?>
                        <div class="row mb-4">
                            <div class="col-md-12">
                                <div class="card">
                                    <div class="card-header">
                                        <h6 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Message Statistics</h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <?php foreach (['sms', 'email', 'whatsapp'] as $type): ?>
                                                <?php if (isset($stats[$type])): ?>
                                                <div class="col-md-4">
                                                    <div class="text-center p-3 border rounded">
                                                        <h5 class="text-capitalize"><?php echo $type; ?></h5>
                                                        <div class="d-flex justify-content-around">
                                                            <div class="text-success">
                                                                <i class="fas fa-check-circle"></i>
                                                                <div class="fw-bold"><?php echo $stats[$type]['success'] ?? 0; ?></div>
                                                                <small>Success</small>
                                                            </div>
                                                            <div class="text-danger">
                                                                <i class="fas fa-times-circle"></i>
                                                                <div class="fw-bold"><?php echo $stats[$type]['failed'] ?? 0; ?></div>
                                                                <small>Failed</small>
                                                            </div>
                                                            <div class="text-warning">
                                                                <i class="fas fa-clock"></i>
                                                                <div class="fw-bold"><?php echo $stats[$type]['pending'] ?? 0; ?></div>
                                                                <small>Pending</small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- Modern Timeline Logs Display -->
                        <div class="timeline-container position-relative ps-4">
                            <div class="timeline-bar position-absolute top-0 start-0 h-100" style="width:4px; background: linear-gradient(180deg, #6366f1 0%, #38bdf8 100%); border-radius:2px;"></div>
                            <?php if (!empty($logs)): ?>
                                <?php foreach ($logs as $log): ?>
                                    <?php
                                        // Color for action type
                                        $badgeClass = 'bg-secondary';
                                        if (stripos($log['action_type'], 'success') !== false) $badgeClass = 'bg-success';
                                        elseif (stripos($log['action_type'], 'fail') !== false || stripos($log['action_type'], 'error') !== false) $badgeClass = 'bg-danger';
                                        elseif (stripos($log['action_type'], 'update') !== false) $badgeClass = 'bg-warning text-dark';
                                        elseif (stripos($log['action_type'], 'sms') !== false) $badgeClass = 'bg-info text-dark';
                                        elseif (stripos($log['action_type'], 'email') !== false) $badgeClass = 'bg-primary';
                                        elseif (stripos($log['action_type'], 'whatsapp') !== false) $badgeClass = 'bg-success';

                                        // Message status badge
                                        $messageStatusClass = '';
                                        $messageStatusIcon = '';
                                        if ($log['message_status'] === 'success') {
                                            $messageStatusClass = 'bg-success';
                                            $messageStatusIcon = 'fas fa-check-circle';
                                        } elseif ($log['message_status'] === 'failed') {
                                            $messageStatusClass = 'bg-danger';
                                            $messageStatusIcon = 'fas fa-times-circle';
                                        } elseif ($log['message_status'] === 'pending') {
                                            $messageStatusClass = 'bg-warning';
                                            $messageStatusIcon = 'fas fa-clock';
                                        }

                                        // Message type icon
                                        $messageTypeIcon = '';
                                        if ($log['message_type'] === 'sms') {
                                            $messageTypeIcon = 'fas fa-sms';
                                        } elseif ($log['message_type'] === 'email') {
                                            $messageTypeIcon = 'fas fa-envelope';
                                        } elseif ($log['message_type'] === 'whatsapp') {
                                            $messageTypeIcon = 'fab fa-whatsapp';
                                        }

                                        // User avatar/initials
                                        $initials = strtoupper(substr($log['username'],0,1));
                                    ?>
                                    <div class="timeline-entry mb-4 position-relative">
                                        <div class="timeline-dot">
                                            <span class="avatar-initials"><?php echo $initials; ?></span>
                                        </div>
                                        <div class="ms-5 flex-grow-1 card-glass">
                                            <div class="d-flex align-items-center mb-1">
                                                <span class="fw-bold text-dark me-2" style="font-size:1.1rem;"><i class="fas fa-user-circle me-1"></i><?php echo htmlspecialchars($log['username']); ?></span>
                                                <span class="badge <?php echo $badgeClass; ?> ms-2"><?php echo ucwords(str_replace('_', ' ', $log['action_type'] ?? $log['action'])); ?></span>
                                                <?php if ($log['message_type']): ?>
                                                    <span class="badge bg-info ms-2"><i class="<?php echo $messageTypeIcon; ?> me-1"></i><?php echo ucfirst($log['message_type']); ?></span>
                                                <?php endif; ?>
                                                <?php if ($log['message_status']): ?>
                                                    <span class="badge <?php echo $messageStatusClass; ?> ms-2"><i class="<?php echo $messageStatusIcon; ?> me-1"></i><?php echo ucfirst($log['message_status']); ?></span>
                                                <?php endif; ?>
                                                <span class="ms-3 text-muted" style="font-size:0.95rem;"><i class="fas fa-clock me-1"></i><?php echo date('M d, Y H:i', strtotime($log['created_at'])); ?></span>
                                            </div>
                                            <div class="mb-1">
                                                <span class="me-3"><i class="fas fa-cube text-primary me-1"></i><span class="fw-semibold text-primary"><?php echo ucfirst($log['module'] ?? 'System'); ?></span></span>
                                                <span><i class="fas fa-tasks text-success me-1"></i><span class="fw-semibold text-success"><?php echo ucwords(str_replace('_', ' ', $log['action'])); ?></span></span>
                                                <?php if ($log['recipient']): ?>
                                                    <span class="me-3"><i class="fas fa-user text-info me-1"></i><span class="fw-semibold text-info"><?php echo htmlspecialchars($log['recipient']); ?></span></span>
                                                <?php endif; ?>
                                            </div>
                                            <div class="log-details">
                                                <?php if ($log['message_type']): ?>
                                                    <div class="mb-2 fw-bold text-primary">
                                                        <?php
                                                            $type = ucfirst($log['message_type']);
                                                            $to = $log['recipient'] ? htmlspecialchars($log['recipient']) : 'Unknown';
                                                            $purpose = '';
                                                            if (stripos($log['action'], 'remind') !== false || stripos($log['action_type'], 'remind') !== false) {
                                                                $purpose = 'Reminder';
                                                            } elseif (stripos($log['action'], 'add') !== false || stripos($log['action_type'], 'add') !== false) {
                                                                $purpose = 'Add Item';
                                                            } else {
                                                                $purpose = ucwords(str_replace('_', ' ', $log['action']));
                                                            }
                                                        ?>
                                                        <?php echo $type; ?> sent to <?php echo $to; ?> — <?php echo $purpose; ?>
                                                    </div>
                                                <?php else: ?>
                                                    <?php if ($log['message_content']): ?>
                                                        <div class="mb-2">
                                                            <strong>Message:</strong><br>
                                                            <div class="bg-light p-2 rounded" style="font-family: monospace; font-size: 0.9rem;">
                                                                <?php echo nl2br(htmlspecialchars($log['message_content'])); ?>
                                                            </div>
                                                        </div>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                                <?php if ($log['error_message']): ?>
                                                    <div class="mb-2">
                                                        <strong class="text-danger">Error:</strong><br>
                                                        <div class="bg-danger bg-opacity-10 p-2 rounded text-danger" style="font-family: monospace; font-size: 0.9rem;">
                                                            <?php echo nl2br(htmlspecialchars($log['error_message'])); ?>
                                                        </div>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if ($log['details']): ?>
                                                    <div>
                                                        <strong>Details:</strong><br>
                                                        <?php echo nl2br(htmlspecialchars($log['details'])); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="alert alert-info mt-4">
                                    <i class="fas fa-info-circle me-2"></i>
                                    No logs found matching your criteria.
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php if ($show_messages): ?>
        <a href="system_logs.php" class="btn btn-secondary mb-3"><i class="fas fa-arrow-left me-2"></i>Back to All Logs</a>
    <?php endif; ?>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <?php 
    // Add permission modal components
    echo getPermissionModalCSS();
    echo getPermissionModalHTML();
    echo getPermissionModalJS();
    ?>
    
    <?php echo getUniversalPermissionModalHTML(); ?>
    <?php echo getUniversalPermissionModalJS(); ?>
    
    <?php include '../includes/footer.php'; ?>
</body>
</html> 